unit XShelLnk;

{
  TShellLink ver. 1.1 (1999-12-06)
  ===================
  A component for Delphi 3/4 to work with shell links - to create and read LNKs.

  Thanks to Radek Voltr (voltr.radek/4600/epr@epr1.ccmail.x400.cez.cz)
  for basic idea.

  Freeware.

  Copyright  1999 Roman Stedronsky, Roman.Stedronsky@seznam.cz.

  All rights reserved. You may use this software in an application
  without fee or royalty, provided this copyright notice remains intact.

  properties:

    property AutoLoad: boolean;			auto load when FileName changed
    property ResolveMode: TResolveMode;		modes for resolving a link
    property ResolveTimeout: word;		timeout duration for nonUI resolving

    property Arguments: string;			command line parameters
    property Description: string;		window title for DOS programs
    property HotKey: TShortCut;			keyboard shortcut
    property FileName: string;			file name of shell link
    property IconFile: string;			icon file name
    property IconIndex: integer;		index of icon
    property LinkName: string;			full path and name of linked file
    property ShowCommand: TWindowState;		window starting status
    property WorkingDirectory: string;		self explaining

  methods:

    procedure Load;				load info from a shell link
    procedure Save;				save info to a shell link
    procedure Resolve;				resolve a shell link

    function GetWindowState;			translate shell link's state to TWindowState
    function KeyToShortCut;			translate shell link's HotKey to TShortCut
    function ShortCutToKey;                     translate TShortCut to shell link's HotKey
}

interface

uses
  Classes, Windows, OLE2, SysUtils, DsgnIntf;//, {$IFDEF VER130} DsIntf{$ENDIF};

type
  TShortCut = Low(Word)..High(Word);
  TResolveModeItems = (rmAnyMatch, rmNoUserInterface, rmUpdate);
  TResolveMode = set of TResolveModeItems;
  TWindowState = (wsNormal, wsMinimized, wsMaximized);
  TIconFile = string;
  TLinkFile = string;

  TShellLink = class(TComponent)
  private
    FAutoLoad: boolean;
    FResolveMode: TResolveMode;
    FResolveTimeout: word;
    FArguments: string;
    FDescription: string;
    FFileName: string;
    FHotKey: TShortCut;
    FIconFile: string;
    FIconIndex: integer;
    FLinkName: string;
    FShowCommand: TWindowState;
    FWorkingDirectory: string;
  protected
    procedure FWriteAutoLoad(Value: boolean); virtual;
    procedure FWriteFileName(Value: string); virtual;
    procedure FWriteIconIndex(Value: integer); virtual;
    procedure FClear; virtual;
  public
    constructor Create(AOwner: TComponent); override;
    function Load: boolean;
    function Save: boolean;
    function Resolve(Handle: Hwnd): boolean;
    // helpful methods
    function GetWindowState(State: integer): TWindowState; virtual;
    function KeyToShortCut(HotKey: word): TShortCut; virtual;
    function ShortCutToKey(ShortCut: TShortCut): Word; virtual;
  published
    property AutoLoad: boolean read FAutoLoad write FWriteAutoLoad default true;
    property ResolveMode: TResolveMode read FResolveMode write FResolveMode default [rmUpdate];
    property ResolveTimeout: word read FResolveTimeout write FResolveTimeout default 3000;
    property Arguments: string read FArguments write FArguments stored false;
    property Description: string read FDescription write FDescription stored false;
    property HotKey: TShortCut read FHotKey write FHotKey stored false;
    property FileName: string read FFileName write FWriteFileName;
    property IconFile: string read FIconFile write FIconFile stored false;
    property IconIndex: integer read FIconIndex write FWriteIconIndex stored false;
    property LinkName: string read FLinkName write FLinkName stored false;
    property ShowCommand: TWindowState read FShowCommand write FShowCommand stored false;
    property WorkingDirectory: string read FWorkingDirectory write FWorkingDirectory stored false;
  end;

  IShellLink = class(IUnknown) { sl }
    function GetPath(pszFile: PAnsiChar; cchMaxPath: Integer; var pfd: TWin32FindData; fFlags: DWORD): HResult; virtual; stdcall; abstract;
    function GetIDList(var ppidl: pointer): HResult; virtual; stdcall; abstract;
    function SetIDList(pidl: pointer): HResult; virtual; stdcall; abstract;
    function GetDescription(pszName: PAnsiChar; cchMaxName: Integer): HResult; virtual; stdcall; abstract;
    function SetDescription(pszName: PAnsiChar): HResult; virtual; stdcall; abstract;
    function GetWorkingDirectory(pszDir: PAnsiChar; cchMaxPath: Integer): HResult; virtual; stdcall; abstract;
    function SetWorkingDirectory(pszDir: PAnsiChar): HResult; virtual; stdcall; abstract;
    function GetArguments(pszArgs: PAnsiChar; cchMaxPath: Integer): HResult; virtual; stdcall; abstract;
    function SetArguments(pszArgs: PAnsiChar): HResult; virtual; stdcall; abstract;
    function GetHotkey(var pwHotkey: Word): HResult; virtual; stdcall; abstract;
    function SetHotkey(wHotkey: Word): HResult; virtual; stdcall; abstract;
    function GetShowCmd(out piShowCmd: Integer): HResult; virtual; stdcall; abstract;
    function SetShowCmd(iShowCmd: Integer): HResult; virtual; stdcall; abstract;
    function GetIconLocation(pszIconPath: PAnsiChar; cchIconPath: Integer; out piIcon: Integer): HResult; virtual; stdcall; abstract;
    function SetIconLocation(pszIconPath: PAnsiChar; iIcon: Integer): HResult; virtual; stdcall; abstract;
    function SetRelativePath(pszPathRel: PAnsiChar; dwReserved: DWORD): HResult; virtual; stdcall; abstract;
    function Resolve(Wnd: HWND; fFlags: DWORD): HResult; virtual; stdcall; abstract;
    function SetPath(pszFile: PAnsiChar): HResult; virtual; stdcall; abstract;
  end;

procedure Register;

implementation

const
  ShowCommands: array[TWindowState] of Integer = (SW_SHOWNORMAL, SW_SHOWMINNOACTIVE, SW_SHOWMAXIMIZED);

  SLR_NO_UI = $0001;
  SLR_ANY_MATCH	 = $0002;
  SLR_UPDATE = $0004;

  SLGP_SHORTPATH = $0001;
  SLGP_UNCPRIORITY = $0002;

  CLSID_ShellLink: TGUID = (D1: $00021401; D2: $0; D3: $0; D4: ($C0, $0, $0, $0, $0, $0, $0, $46));
  IID_IShellLink: TGUID = (D1: $000214EE; D2: $0000; D3: $0000; D4: ($C0, $00, $00, $00, $00, $00, $00, $46));

  scShift = $2000;
  scCtrl = $4000;
  scAlt = $8000;
  scNone = 0;

  HOTKEYF_SHIFT = $01;
  HOTKEYF_CONTROL = $02;
  HOTKEYF_ALT = $04;
  HOTKEYF_EXT = $08;

constructor TShellLink.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  AutoLoad := true;
  ResolveMode := [rmUpdate];
  ResolveTimeout := 3000;
end;

procedure TShellLink.FWriteAutoLoad(Value: boolean);
begin
  if FAutoLoad <> Value then
  begin
    FAutoLoad := Value;
    if FAutoLoad = true then
      Load;
  end;
end;

procedure TShellLink.FWriteIconIndex(Value: integer);
begin
  if Value >= 0 then
    FIconIndex := Value;
end;

procedure TShellLink.FWriteFileName(Value: string);
begin
  if Value <> FFileName then
  begin
    FFileName := Value;
    if FAutoLoad then
      Load;
  end;
end;

procedure TShellLink.FClear;
begin
  FArguments := '';
  FDescription := '';
  FHotKey  := 0;
  FIconFile  := '';
  FIconIndex  := 0;
  FLinkName := '';
  FShowCommand := wsNormal;
  FWorkingDirectory  := '';
end;

function TShellLink.Load: boolean;
var
  Str: array[0..MAX_PATH] of Char;
  Index: integer;
  ShellLink: IShellLink;
  PersistFile: IPersistFile;
  FindData: TWin32FindData;
begin
  Result := false;
  CoInitialize(nil);
  if Succeeded(COCreateInstance(CLSID_ShellLink, nil, CLSCTX_INPROC_SERVER, IID_IShellLink, ShellLink)) then
  begin
    if Succeeded(ShellLink.QueryInterface(IID_IPersistFile, PersistFile)) then
    begin
      StringToWideChar(FFileName, @Str, SizeOf(Str));
      if PersistFile.Load(@Str, STGM_READ) = NOERROR then
      begin;
        ShellLink.GetArguments(Str, MAX_PATH);
        FArguments := Str;
        ShellLink.GetDescription(Str, MAX_PATH);
        FDescription := Str;
        ShellLink.GetHotKey(Word(FHotKey));
        FHotKey := KeyToShortCut(FHotKey);
        ShellLink.GetIconLocation(Str, MAX_PATH, Index);
        FIconFile := Str;
        FIconIndex := Index;
        ShellLink.GetPath(Str, MAX_PATH, FindData, SLGP_UNCPRIORITY);
        FLinkName := Str;
        ShellLink.GetShowCmd(Index);
        FShowCommand := GetWindowState(Index);
        ShellLink.GetWorkingDirectory(Str, MAX_PATH);
        FWorkingDirectory := Str;
        Result := true;
      end;
    end;
    PersistFile.Release;
    ShellLink.Release;
    CoUninitialize;
  end;
  if not Result then
    FClear;
end;

function TShellLink.Save: boolean;
var
  Str: array [0..MAX_PATH] of Char;
  ShellLink: IShellLink;
  PersistFile: IPersistFile;
begin
  Result := false;
  CoInitialize(nil);
  if Succeeded(COCreateInstance(CLSID_ShellLink, nil, CLSCTX_INPROC_SERVER, IID_IShellLink, ShellLink)) then
  begin
    if Succeeded(ShellLink.QueryInterface(IID_IPersistFile, PersistFile)) then
    begin
      StrPCopy(@Str, FArguments);
      ShellLink.SetArguments(Str);
      StrPCopy(@Str, FDescription);
      ShellLink.SetDescription(Str);
      ShellLink.SetHotKey(ShortCutToKey(FHotKey));
      StrPCopy(@Str, FIconFile);
      ShellLink.SetIconLocation(Str, FIconIndex);
      StrPCopy(@Str, FLinkName);
      ShellLink.SetPath(Str);
      ShellLink.SetShowCmd(ShowCommands[FShowCommand]);
      StrPCopy(@Str, FWorkingDirectory);
      ShellLink.SetWorkingDirectory(Str);
      StringToWideChar(FFileName, @Str, SizeOf(Str));
      if PersistFile.Save(@Str, false) = NOERROR then
        Result := true;
    end;
    PersistFile.Release;
    ShellLink.Release;
    CoUninitialize;
  end;
end;

function TShellLink.Resolve(Handle: Hwnd): boolean;
var
  Str: array[0..MAX_PATH] of Char;
  ShellLink: IShellLink;
  PersistFile: IPersistFile;
  function GetResolveMode: word;
  begin
    Result := 0;
    if rmNoUserInterface in FResolveMode then Result := Result or SLR_NO_UI;
    if rmAnyMatch in FResolveMode then Result := Result or SLR_ANY_MATCH;
    if rmUpdate in FResolveMode then Result := Result or SLR_UPDATE;
  end;
begin
  Result := false;
  CoInitialize(nil);
  if Succeeded(COCreateInstance(CLSID_ShellLink, nil, CLSCTX_INPROC_SERVER, IID_IShellLink, ShellLink)) then
  begin
    if Succeeded(ShellLink.QueryInterface(IID_IPersistFile, PersistFile)) then
    begin
      StringToWideChar(FFileName, @Str, SizeOf(Str));
      PersistFile.Load(@Str, STGM_READ);
      Result := ShellLink.Resolve(Handle, MakeLong(GetResolveMode, FResolveTimeout)) = NOERROR;
    end;
    PersistFile.Release;
    ShellLink.Release;
    CoUninitialize;
  end;
end;

function TShellLink.GetWindowState(State: integer): TWindowState;
begin
  case State of
    SW_SHOWNORMAL, SW_SHOW, SW_RESTORE: Result := wsNormal;
    SW_SHOWMINIMIZED, SW_MINIMIZE, SW_SHOWMINNOACTIVE: Result := wsMinimized;
    SW_SHOWMAXIMIZED, SW_MAX: Result := wsMaximized;
  else
    Result := wsNormal;
  end;
end;

function TShellLink.KeyToShortCut(HotKey: word): TShortCut;
begin
  Result := WordRec(HotKey).Lo;
  if ((HotKey shr 8) and HOTKEYF_SHIFT) <> 0 then Inc(Result, scShift);
  if ((HotKey shr 8) and HOTKEYF_CONTROL) <> 0 then Inc(Result, scCtrl);
  if ((HotKey shr 8) and HOTKEYF_ALT) <> 0 then Inc(Result, scAlt);
end;

function TShellLink.ShortCutToKey(ShortCut: TShortCut): Word;
var
  Key: byte;
  Shift: byte;
begin
  Key := ShortCut and not (scShift + scCtrl + scAlt);
  Shift := 0;
  if ShortCut and scShift <> 0 then Inc(Shift, HOTKEYF_SHIFT);
  if ShortCut and scCtrl <> 0 then Inc(Shift, HOTKEYF_CONTROL);
  if ShortCut and scAlt <> 0 then Inc(Shift, HOTKEYF_ALT);
  Result := MakeWord(Key, Shift);
end;

procedure Register;
begin
  RegisterComponents('Extra', [TShellLink]);
  RegisterPropertyEditor(TypeInfo(TShortCut), nil, '', TShortCutProperty);
end;

end.
